/*
 * Decompiled with CFR 0.152.
 */
package net.sf.jabref.shared;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import net.sf.jabref.logic.exporter.BibDatabaseWriter;
import net.sf.jabref.logic.exporter.MetaDataSerializer;
import net.sf.jabref.logic.importer.ParseException;
import net.sf.jabref.logic.importer.util.MetaDataParser;
import net.sf.jabref.model.bibtexkeypattern.GlobalBibtexKeyPattern;
import net.sf.jabref.model.database.BibDatabase;
import net.sf.jabref.model.database.BibDatabaseContext;
import net.sf.jabref.model.database.event.EntryAddedEvent;
import net.sf.jabref.model.database.event.EntryRemovedEvent;
import net.sf.jabref.model.entry.BibEntry;
import net.sf.jabref.model.entry.event.EntryEvent;
import net.sf.jabref.model.entry.event.EntryEventSource;
import net.sf.jabref.model.entry.event.FieldChangedEvent;
import net.sf.jabref.model.metadata.MetaData;
import net.sf.jabref.model.metadata.event.MetaDataChangedEvent;
import net.sf.jabref.shared.DBMSConnection;
import net.sf.jabref.shared.DBMSConnectionProperties;
import net.sf.jabref.shared.DBMSProcessor;
import net.sf.jabref.shared.DBMSType;
import net.sf.jabref.shared.event.ConnectionLostEvent;
import net.sf.jabref.shared.event.SharedEntryNotPresentEvent;
import net.sf.jabref.shared.event.UpdateRefusedEvent;
import net.sf.jabref.shared.exception.DatabaseNotSupportedException;
import net.sf.jabref.shared.exception.InvalidDBMSConnectionPropertiesException;
import net.sf.jabref.shared.exception.OfflineLockException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class DBMSSynchronizer {
    private static final Log LOGGER = LogFactory.getLog(DBMSSynchronizer.class);
    private DBMSProcessor dbmsProcessor;
    private DBMSType dbmsType;
    private String dbName;
    private final BibDatabaseContext bibDatabaseContext;
    private MetaData metaData;
    private final BibDatabase bibDatabase;
    private final EventBus eventBus;
    private Connection currentConnection;
    private final Character keywordSeparator;
    private GlobalBibtexKeyPattern globalCiteKeyPattern;

    public DBMSSynchronizer(BibDatabaseContext bibDatabaseContext, Character keywordSeparator, GlobalBibtexKeyPattern globalCiteKeyPattern) {
        this.bibDatabaseContext = Objects.requireNonNull(bibDatabaseContext);
        this.bibDatabase = bibDatabaseContext.getDatabase();
        this.metaData = bibDatabaseContext.getMetaData();
        this.eventBus = new EventBus();
        this.keywordSeparator = keywordSeparator;
        this.globalCiteKeyPattern = Objects.requireNonNull(globalCiteKeyPattern);
    }

    @Subscribe
    public void listen(EntryAddedEvent event) {
        if (this.isEventSourceAccepted(event) && this.checkCurrentConnection()) {
            this.dbmsProcessor.insertEntry(event.getBibEntry());
            this.synchronizeLocalMetaData();
            this.synchronizeLocalDatabase();
        }
    }

    @Subscribe
    public void listen(FieldChangedEvent event) {
        if (this.isPresentLocalBibEntry(event.getBibEntry()) && this.isEventSourceAccepted(event) && this.checkCurrentConnection()) {
            this.synchronizeLocalMetaData();
            BibEntry bibEntry = event.getBibEntry();
            this.synchronizeSharedEntry(bibEntry);
            this.synchronizeLocalDatabase();
        }
    }

    @Subscribe
    public void listen(EntryRemovedEvent event) {
        if (this.isEventSourceAccepted(event) && this.checkCurrentConnection()) {
            this.dbmsProcessor.removeEntry(event.getBibEntry());
            this.synchronizeLocalMetaData();
            this.synchronizeLocalDatabase();
        }
    }

    @Subscribe
    public void listen(MetaDataChangedEvent event) {
        if (this.checkCurrentConnection()) {
            this.synchronizeSharedMetaData(event.getMetaData(), this.globalCiteKeyPattern);
            this.synchronizeLocalDatabase();
            this.applyMetaData();
            this.dbmsProcessor.notifyClients();
        }
    }

    @Subscribe
    public void listen(EntryEvent event) {
        if (this.isEventSourceAccepted(event)) {
            this.dbmsProcessor.notifyClients();
        }
    }

    public void initializeDatabases() throws DatabaseNotSupportedException, SQLException {
        if (!this.dbmsProcessor.checkBaseIntegrity()) {
            LOGGER.info("Integrity check failed. Fixing...");
            this.dbmsProcessor.setupSharedDatabase();
            if (this.dbmsProcessor.checkForPre3Dot6Intergrity()) {
                throw new DatabaseNotSupportedException();
            }
        }
        this.synchronizeLocalMetaData();
        this.synchronizeLocalDatabase();
        this.dbmsProcessor.startNotificationListener(this);
    }

    public void synchronizeLocalDatabase() {
        if (!this.checkCurrentConnection()) {
            return;
        }
        List<BibEntry> localEntries = this.bibDatabase.getEntries();
        Map<Integer, Integer> idVersionMap = this.dbmsProcessor.getSharedIDVersionMapping();
        this.removeNotSharedEntries(localEntries, idVersionMap.keySet());
        for (Map.Entry<Integer, Integer> idVersionEntry : idVersionMap.entrySet()) {
            Optional<BibEntry> bibEntry;
            boolean match = false;
            for (BibEntry localEntry : localEntries) {
                Optional<BibEntry> sharedEntry;
                if (idVersionEntry.getKey().intValue() != localEntry.getSharedBibEntryData().getSharedID()) continue;
                match = true;
                if (idVersionEntry.getValue() <= localEntry.getSharedBibEntryData().getVersion() || !(sharedEntry = this.dbmsProcessor.getSharedEntry(idVersionEntry.getKey())).isPresent()) continue;
                localEntry.setType(sharedEntry.get().getType(), EntryEventSource.SHARED);
                localEntry.getSharedBibEntryData().setVersion(sharedEntry.get().getSharedBibEntryData().getVersion());
                for (String field : sharedEntry.get().getFieldNames()) {
                    localEntry.setField(field, sharedEntry.get().getField(field), EntryEventSource.SHARED);
                }
                Set<String> redundantLocalEntryFields = localEntry.getFieldNames();
                redundantLocalEntryFields.removeAll(sharedEntry.get().getFieldNames());
                for (String redundantField : redundantLocalEntryFields) {
                    localEntry.clearField(redundantField, EntryEventSource.SHARED);
                }
            }
            if (match || !(bibEntry = this.dbmsProcessor.getSharedEntry(idVersionEntry.getKey())).isPresent()) continue;
            this.bibDatabase.insertEntry(bibEntry.get(), EntryEventSource.SHARED);
        }
    }

    private void removeNotSharedEntries(List<BibEntry> localEntries, Set<Integer> sharedIDs) {
        for (int i = 0; i < localEntries.size(); ++i) {
            BibEntry localEntry = localEntries.get(i);
            boolean match = false;
            for (int sharedID : sharedIDs) {
                if (localEntry.getSharedBibEntryData().getSharedID() != sharedID) continue;
                match = true;
                break;
            }
            if (match) continue;
            this.eventBus.post(new SharedEntryNotPresentEvent(localEntry));
            this.bibDatabase.removeEntry(localEntry, EntryEventSource.SHARED);
            --i;
        }
    }

    public void synchronizeSharedEntry(BibEntry bibEntry) {
        if (!this.checkCurrentConnection()) {
            return;
        }
        try {
            BibDatabaseWriter.applySaveActions(bibEntry, this.metaData);
            this.dbmsProcessor.updateEntry(bibEntry);
        }
        catch (OfflineLockException exception) {
            this.eventBus.post(new UpdateRefusedEvent(this.bibDatabaseContext, exception.getLocalBibEntry(), exception.getSharedBibEntry()));
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error: ", e);
        }
    }

    public void synchronizeLocalMetaData() {
        if (!this.checkCurrentConnection()) {
            return;
        }
        try {
            this.metaData = MetaDataParser.parse(this.dbmsProcessor.getSharedMetaData(), this.keywordSeparator);
        }
        catch (ParseException e) {
            LOGGER.error("Parse error", e);
        }
    }

    private void synchronizeSharedMetaData(MetaData data, GlobalBibtexKeyPattern globalCiteKeyPattern) {
        if (!this.checkCurrentConnection()) {
            return;
        }
        try {
            this.dbmsProcessor.setSharedMetaData(MetaDataSerializer.getSerializedStringMap(data, globalCiteKeyPattern));
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error: ", e);
        }
    }

    public void applyMetaData() {
        if (!this.checkCurrentConnection()) {
            return;
        }
        for (BibEntry bibEntry : this.bibDatabase.getEntries()) {
            if (BibDatabaseWriter.applySaveActions(bibEntry, this.metaData).isEmpty()) continue;
            try {
                this.dbmsProcessor.updateEntry(bibEntry);
            }
            catch (OfflineLockException exception) {
                this.eventBus.post(new UpdateRefusedEvent(this.bibDatabaseContext, exception.getLocalBibEntry(), exception.getSharedBibEntry()));
            }
            catch (SQLException e) {
                LOGGER.error("SQL Error: ", e);
            }
        }
    }

    public void pullChanges() {
        if (!this.checkCurrentConnection()) {
            return;
        }
        this.synchronizeLocalDatabase();
        this.synchronizeLocalMetaData();
    }

    public boolean checkCurrentConnection() {
        try {
            boolean isValid = this.currentConnection.isValid(0);
            if (!isValid) {
                this.eventBus.post(new ConnectionLostEvent(this.bibDatabaseContext));
            }
            return isValid;
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error:", e);
            return false;
        }
    }

    public boolean isEventSourceAccepted(EntryEvent event) {
        EntryEventSource eventSource = event.getEntryEventSource();
        return eventSource == EntryEventSource.LOCAL || eventSource == EntryEventSource.UNDO;
    }

    public void openSharedDatabase(DBMSConnection connection) throws DatabaseNotSupportedException, SQLException {
        this.dbmsType = connection.getProperties().getType();
        this.dbName = connection.getProperties().getDatabase();
        this.currentConnection = connection.getConnection();
        this.dbmsProcessor = DBMSProcessor.getProcessorInstance(connection);
        this.initializeDatabases();
    }

    public void openSharedDatabase(DBMSConnectionProperties properties) throws SQLException, DatabaseNotSupportedException, InvalidDBMSConnectionPropertiesException {
        this.openSharedDatabase(new DBMSConnection(properties));
    }

    public void closeSharedDatabase() {
        try {
            this.dbmsProcessor.stopNotificationListener();
            this.currentConnection.close();
        }
        catch (SQLException e) {
            LOGGER.error("SQL Error:", e);
        }
    }

    private boolean isPresentLocalBibEntry(BibEntry bibEntry) {
        return this.bibDatabase.getEntries().contains(bibEntry);
    }

    public String getDBName() {
        return this.dbName;
    }

    public DBMSType getDBType() {
        return this.dbmsType;
    }

    public DBMSProcessor getDBProcessor() {
        return this.dbmsProcessor;
    }

    public void setMetaData(MetaData metaData) {
        this.metaData = metaData;
    }

    public void registerListener(Object listener) {
        this.eventBus.register(listener);
    }
}

