/*
 * Decompiled with CFR 0.152.
 */
package net.sf.jabref.model.entry;

import com.google.common.base.Strings;
import com.google.common.eventbus.EventBus;
import java.text.DateFormat;
import java.text.FieldPosition;
import java.text.ParseException;
import java.text.ParsePosition;
import java.text.SimpleDateFormat;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;
import net.sf.jabref.model.EntryTypes;
import net.sf.jabref.model.FieldChange;
import net.sf.jabref.model.database.BibDatabase;
import net.sf.jabref.model.database.BibDatabaseMode;
import net.sf.jabref.model.entry.CanonicalBibtexEntry;
import net.sf.jabref.model.entry.EntryConverter;
import net.sf.jabref.model.entry.EntryLinkList;
import net.sf.jabref.model.entry.EntryType;
import net.sf.jabref.model.entry.IdGenerator;
import net.sf.jabref.model.entry.Keyword;
import net.sf.jabref.model.entry.KeywordList;
import net.sf.jabref.model.entry.MonthUtil;
import net.sf.jabref.model.entry.ParsedEntryLink;
import net.sf.jabref.model.entry.SharedBibEntryData;
import net.sf.jabref.model.entry.event.EntryEventSource;
import net.sf.jabref.model.entry.event.FieldChangedEvent;
import net.sf.jabref.model.strings.LatexToUnicode;
import net.sf.jabref.model.strings.StringUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class BibEntry
implements Cloneable {
    private static final Log LOGGER = LogFactory.getLog(BibEntry.class);
    public static final String TYPE_HEADER = "entrytype";
    public static final String OBSOLETE_TYPE_HEADER = "bibtextype";
    public static final String KEY_FIELD = "bibtexkey";
    protected static final String ID_FIELD = "id";
    public static final String DEFAULT_TYPE = "misc";
    private static final Pattern REMOVE_TRAILING_WHITESPACE = Pattern.compile("\\s+$");
    private String id;
    private final SharedBibEntryData sharedBibEntryData;
    private String type;
    private Map<String, String> fields = new ConcurrentHashMap<String, String>();
    private final Map<String, Set<String>> fieldsAsWords = new HashMap<String, Set<String>>();
    private final Map<String, String> latexFreeFields = new ConcurrentHashMap<String, String>();
    private LatexToUnicode unicodeConverter = new LatexToUnicode();
    private boolean searchHit;
    private boolean groupHit;
    private String parsedSerialization;
    private String commentsBeforeEntry = "";
    private boolean changed;
    private final EventBus eventBus = new EventBus();

    public BibEntry() {
        this(IdGenerator.next());
    }

    public BibEntry(String id) {
        this(id, DEFAULT_TYPE);
    }

    public BibEntry(String id, String type) {
        Objects.requireNonNull(id, "Every BibEntry must have an ID");
        this.id = id;
        this.setType(type);
        this.sharedBibEntryData = new SharedBibEntryData();
    }

    public Optional<FieldChange> replaceKeywords(KeywordList keywordsToReplace, Optional<Keyword> newValue, Character keywordDelimiter) {
        KeywordList keywordList = this.getKeywords(keywordDelimiter);
        keywordList.replaceKeywords(keywordsToReplace, newValue);
        return this.putKeywords(keywordList, keywordDelimiter);
    }

    public Optional<String> getResolvedFieldOrAlias(String field, BibDatabase database) {
        Objects.requireNonNull(this, "entry cannot be null");
        if (TYPE_HEADER.equals(field) || OBSOLETE_TYPE_HEADER.equals(field)) {
            Optional<EntryType> entryType = EntryTypes.getType(this.getType(), BibDatabaseMode.BIBLATEX);
            if (entryType.isPresent()) {
                return Optional.of(entryType.get().getName());
            }
            return Optional.of(StringUtil.capitalizeFirst(this.getType()));
        }
        if (KEY_FIELD.equals(field)) {
            return this.getCiteKeyOptional();
        }
        Optional<String> result = this.getFieldOrAlias(field);
        if (!result.isPresent() && database != null) {
            Optional<BibEntry> referred = database.getReferencedEntry(this);
            result = referred.flatMap(entry -> entry.getFieldOrAlias(field));
        }
        return result.map(resultText -> BibDatabase.getText(resultText, database));
    }

    public void setId(String id) {
        Objects.requireNonNull(id, "Every BibEntry must have an ID");
        String oldId = this.id;
        this.eventBus.post(new FieldChangedEvent(this, ID_FIELD, id, oldId));
        this.id = id;
        this.changed = true;
    }

    public String getId() {
        return this.id;
    }

    public void setCiteKey(String newCiteKey) {
        this.setField(KEY_FIELD, newCiteKey);
    }

    @Deprecated
    public String getCiteKey() {
        return this.fields.get(KEY_FIELD);
    }

    public Optional<String> getCiteKeyOptional() {
        return Optional.ofNullable(this.fields.get(KEY_FIELD));
    }

    public boolean hasCiteKey() {
        return !Strings.isNullOrEmpty(this.getCiteKey());
    }

    public String getType() {
        return this.type;
    }

    public void setType(String type, EntryEventSource eventSource) {
        String newType = Strings.isNullOrEmpty(type) ? DEFAULT_TYPE : type;
        String oldType = this.getField(TYPE_HEADER).orElse(null);
        this.type = newType.toLowerCase(Locale.ENGLISH);
        this.changed = true;
        this.eventBus.post(new FieldChangedEvent(this, TYPE_HEADER, newType, oldType, eventSource));
    }

    public void setType(String type) {
        this.setType(type, EntryEventSource.LOCAL);
    }

    public void setType(EntryType type) {
        this.setType(type.getName());
    }

    public Set<String> getFieldNames() {
        return new TreeSet<String>(this.fields.keySet());
    }

    public Optional<String> getField(String name) {
        return Optional.ofNullable(this.fields.get(this.toLowerCase(name)));
    }

    public boolean hasField(String name) {
        return this.fields.containsKey(this.toLowerCase(name));
    }

    private String toLowerCase(String fieldName) {
        Objects.requireNonNull(fieldName, "field name must not be null");
        return fieldName.toLowerCase(Locale.ENGLISH);
    }

    private Optional<String> genericGetFieldOrAlias(String name, GetFieldInterface getFieldInterface) {
        Optional<String> year;
        Optional<String> fieldValue = getFieldInterface.getValueForField(this.toLowerCase(name));
        if (fieldValue.isPresent() && !fieldValue.get().isEmpty()) {
            return fieldValue;
        }
        String aliasForField = EntryConverter.FIELD_ALIASES.get(name);
        if (aliasForField != null) {
            return getFieldInterface.getValueForField(aliasForField);
        }
        if ("date".equals(name) && (year = getFieldInterface.getValueForField("year")).isPresent()) {
            MonthUtil.Month month = MonthUtil.getMonth(getFieldInterface.getValueForField("month").orElse(""));
            if (month.isValid()) {
                return Optional.of(year.get() + '-' + month.twoDigitNumber);
            }
            return year;
        }
        if ("year".equals(name) || "month".equals(name)) {
            Optional<String> date = getFieldInterface.getValueForField("date");
            if (!date.isPresent()) {
                return Optional.empty();
            }
            DateFormat df = new DateFormat(){
                static final String FORMAT1 = "yyyy-MM-dd";
                static final String FORMAT2 = "yyyy-MM";
                final SimpleDateFormat sdf1 = new SimpleDateFormat("yyyy-MM-dd");
                final SimpleDateFormat sdf2 = new SimpleDateFormat("yyyy-MM");

                @Override
                public StringBuffer format(Date dDate, StringBuffer toAppendTo, FieldPosition fieldPosition) {
                    throw new UnsupportedOperationException();
                }

                @Override
                public Date parse(String source, ParsePosition pos) {
                    if (source.length() - pos.getIndex() == FORMAT1.length()) {
                        return this.sdf1.parse(source, pos);
                    }
                    return this.sdf2.parse(source, pos);
                }
            };
            try {
                Date parsedDate = df.parse(date.get());
                Calendar calendar = Calendar.getInstance();
                calendar.setTime(parsedDate);
                if ("year".equals(name)) {
                    return Optional.of(Integer.toString(calendar.get(1)));
                }
                if ("month".equals(name)) {
                    return Optional.of(Integer.toString(calendar.get(2) + 1));
                }
            }
            catch (ParseException e) {
                df = new SimpleDateFormat("yyyy");
                try {
                    Date parsedDate = df.parse(date.get());
                    Calendar calendar = Calendar.getInstance();
                    calendar.setTime(parsedDate);
                    if ("year".equals(name)) {
                        return Optional.of(Integer.toString(calendar.get(1)));
                    }
                }
                catch (ParseException e2) {
                    LOGGER.warn("Could not parse entry " + name, e2);
                    return Optional.empty();
                }
            }
        }
        return Optional.empty();
    }

    public Optional<String> getFieldOrAliasLatexFree(String name) {
        return this.genericGetFieldOrAlias(name, this::getLatexFreeField);
    }

    public Optional<String> getFieldOrAlias(String name) {
        return this.genericGetFieldOrAlias(name, this::getField);
    }

    public void setField(Map<String, String> fields) {
        Objects.requireNonNull(fields, "fields must not be null");
        fields.forEach(this::setField);
    }

    public Optional<FieldChange> setField(String name, String value, EntryEventSource eventSource) {
        Objects.requireNonNull(name, "field name must not be null");
        Objects.requireNonNull(value, "field value must not be null");
        String fieldName = this.toLowerCase(name);
        if (value.isEmpty()) {
            return this.clearField(fieldName);
        }
        String oldValue = this.getField(fieldName).orElse(null);
        if (value.equals(oldValue)) {
            return Optional.empty();
        }
        if (ID_FIELD.equals(fieldName)) {
            throw new IllegalArgumentException("The field name '" + name + "' is reserved");
        }
        this.changed = true;
        this.fields.put(fieldName, value.intern());
        this.invalidateFieldCache(fieldName);
        FieldChange change = new FieldChange(this, fieldName, oldValue, value);
        this.eventBus.post(new FieldChangedEvent(change, eventSource));
        return Optional.of(change);
    }

    public Optional<FieldChange> setField(String name, Optional<String> value, EntryEventSource eventSource) {
        if (value.isPresent()) {
            return this.setField(name, value.get(), eventSource);
        }
        return Optional.empty();
    }

    public Optional<FieldChange> setField(String name, String value) {
        return this.setField(name, value, EntryEventSource.LOCAL);
    }

    public Optional<FieldChange> clearField(String name) {
        return this.clearField(name, EntryEventSource.LOCAL);
    }

    public Optional<FieldChange> clearField(String name, EntryEventSource eventSource) {
        String fieldName = this.toLowerCase(name);
        if (ID_FIELD.equals(fieldName)) {
            throw new IllegalArgumentException("The field name '" + name + "' is reserved");
        }
        Optional<String> oldValue = this.getField(fieldName);
        if (!oldValue.isPresent()) {
            return Optional.empty();
        }
        this.changed = true;
        this.fields.remove(fieldName);
        this.invalidateFieldCache(fieldName);
        FieldChange change = new FieldChange(this, fieldName, oldValue.get(), null);
        this.eventBus.post(new FieldChangedEvent(change, eventSource));
        return Optional.of(change);
    }

    public boolean allFieldsPresent(List<String> allFields, BibDatabase database) {
        for (String field : allFields) {
            String[] altFields;
            String fieldName = this.toLowerCase(field);
            if (!(fieldName.contains("/") ? !this.atLeastOnePresent(altFields = field.split("/"), database) : !this.getResolvedFieldOrAlias(fieldName, database).isPresent())) continue;
            return false;
        }
        return true;
    }

    private boolean atLeastOnePresent(String[] fieldsToCheck, BibDatabase database) {
        for (String field : fieldsToCheck) {
            String fieldName = this.toLowerCase(field);
            Optional<String> value = this.getResolvedFieldOrAlias(fieldName, database);
            if (!value.isPresent() || value.get().isEmpty()) continue;
            return true;
        }
        return false;
    }

    public Object clone() {
        BibEntry clone = new BibEntry(this.id, this.type);
        clone.fields = new HashMap<String, String>(this.fields);
        return clone;
    }

    public String toString() {
        return CanonicalBibtexEntry.getCanonicalRepresentation(this);
    }

    public boolean isSearchHit() {
        return this.searchHit;
    }

    public void setSearchHit(boolean searchHit) {
        this.searchHit = searchHit;
    }

    public boolean isGroupHit() {
        return this.groupHit;
    }

    public void setGroupHit(boolean groupHit) {
        this.groupHit = groupHit;
    }

    public String getAuthorTitleYear(int maxCharacters) {
        String[] s = new String[]{this.getField("author").orElse("N/A"), this.getField("title").orElse("N/A"), this.getField("year").orElse("N/A")};
        String text = s[0] + ": \"" + s[1] + "\" (" + s[2] + ')';
        if (maxCharacters <= 0 || text.length() <= maxCharacters) {
            return text;
        }
        return text.substring(0, maxCharacters + 1) + "...";
    }

    public Optional<String> getPublicationDate() {
        MonthUtil.Month month;
        if (!this.hasField("year")) {
            return Optional.empty();
        }
        Optional<String> year = this.getField("year");
        Optional<String> monthString = this.getField("month");
        if (monthString.isPresent() && (month = MonthUtil.getMonth(monthString.get())).isValid()) {
            return Optional.of(year.orElse("") + "-" + month.twoDigitNumber);
        }
        return year;
    }

    public void setParsedSerialization(String parsedSerialization) {
        this.changed = false;
        this.parsedSerialization = parsedSerialization;
    }

    public String getParsedSerialization() {
        return this.parsedSerialization;
    }

    public void setCommentsBeforeEntry(String parsedComments) {
        this.commentsBeforeEntry = parsedComments;
    }

    public boolean hasChanged() {
        return this.changed;
    }

    public void setChanged(boolean changed) {
        this.changed = changed;
    }

    public Optional<FieldChange> putKeywords(List<String> keywords, Character delimiter) {
        Objects.requireNonNull(delimiter);
        return this.putKeywords(new KeywordList(keywords), delimiter);
    }

    public Optional<FieldChange> putKeywords(KeywordList keywords, Character delimiter) {
        Objects.requireNonNull(keywords);
        Optional<String> oldValue = this.getField("keywords");
        if (keywords.isEmpty()) {
            if (oldValue.isPresent()) {
                return this.clearField("keywords");
            }
            return Optional.empty();
        }
        String newValue = keywords.getAsString(delimiter);
        return this.setField("keywords", newValue);
    }

    public void addKeyword(String keyword, Character delimiter) {
        Objects.requireNonNull(keyword, "keyword must not be null");
        if (keyword.isEmpty()) {
            return;
        }
        this.addKeyword(new Keyword(keyword), delimiter);
    }

    public void addKeyword(Keyword keyword, Character delimiter) {
        KeywordList keywords = this.getKeywords(delimiter);
        keywords.add(keyword);
        this.putKeywords(keywords, delimiter);
    }

    public void addKeywords(Collection<String> keywords, Character delimiter) {
        Objects.requireNonNull(keywords);
        keywords.forEach(keyword -> this.addKeyword((String)keyword, delimiter));
    }

    public KeywordList getKeywords(Character delimiter) {
        Optional<String> keywordsContent = this.getField("keywords");
        if (keywordsContent.isPresent()) {
            return KeywordList.parse(keywordsContent.get(), delimiter);
        }
        return new KeywordList();
    }

    public Collection<String> getFieldValues() {
        return this.fields.values();
    }

    public Map<String, String> getFieldMap() {
        return this.fields;
    }

    public SharedBibEntryData getSharedBibEntryData() {
        return this.sharedBibEntryData;
    }

    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || this.getClass() != o.getClass()) {
            return false;
        }
        BibEntry entry = (BibEntry)o;
        return Objects.equals(this.type, entry.type) && Objects.equals(this.fields, entry.fields);
    }

    public int hashCode() {
        return Objects.hash(this.type, this.fields);
    }

    public void registerListener(Object object) {
        this.eventBus.register(object);
    }

    public void unregisterListener(Object object) {
        this.eventBus.unregister(object);
    }

    public BibEntry withField(String field, String value) {
        this.setField(field, value);
        return this;
    }

    public String getUserComments() {
        return REMOVE_TRAILING_WHITESPACE.matcher(this.commentsBeforeEntry).replaceFirst("");
    }

    public List<ParsedEntryLink> getEntryLinkList(String fieldName, BibDatabase database) {
        return this.getField(fieldName).map(fieldValue -> EntryLinkList.parse(fieldValue, database)).orElse(Collections.emptyList());
    }

    public Optional<FieldChange> setEntryLinkList(String fieldName, List<ParsedEntryLink> list) {
        return this.setField(fieldName, EntryLinkList.serialize(list));
    }

    public Set<String> getFieldAsWords(String field) {
        String fieldName = this.toLowerCase(field);
        Set<String> storedList = this.fieldsAsWords.get(fieldName);
        if (storedList != null) {
            return storedList;
        }
        String fieldValue = this.fields.get(fieldName);
        if (fieldValue == null) {
            return Collections.emptySet();
        }
        HashSet<String> words = new HashSet<String>(StringUtil.getStringAsWords(fieldValue));
        this.fieldsAsWords.put(fieldName, words);
        return words;
    }

    public Optional<FieldChange> clearCiteKey() {
        return this.clearField(KEY_FIELD);
    }

    private void invalidateFieldCache(String fieldName) {
        this.latexFreeFields.remove(fieldName);
        this.fieldsAsWords.remove(fieldName);
    }

    public Optional<String> getLatexFreeField(String name) {
        if (!this.hasField(name)) {
            return Optional.empty();
        }
        if (this.latexFreeFields.containsKey(name)) {
            return Optional.ofNullable(this.latexFreeFields.get(this.toLowerCase(name)));
        }
        String latexFreeField = this.unicodeConverter.format(this.getField(name).get()).intern();
        this.latexFreeFields.put(name, latexFreeField);
        return Optional.of(latexFreeField);
    }

    private static interface GetFieldInterface {
        public Optional<String> getValueForField(String var1);
    }
}

