/*
 * Decompiled with CFR 0.152.
 */
package net.sf.jabref.model.database;

import com.google.common.eventbus.EventBus;
import com.google.common.eventbus.Subscribe;
import java.math.BigInteger;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import net.sf.jabref.model.EntryTypes;
import net.sf.jabref.model.database.BibDatabaseMode;
import net.sf.jabref.model.database.DuplicationChecker;
import net.sf.jabref.model.database.EntrySorter;
import net.sf.jabref.model.database.KeyChangeListener;
import net.sf.jabref.model.database.KeyCollisionException;
import net.sf.jabref.model.database.event.EntryAddedEvent;
import net.sf.jabref.model.database.event.EntryRemovedEvent;
import net.sf.jabref.model.entry.BibEntry;
import net.sf.jabref.model.entry.BibtexString;
import net.sf.jabref.model.entry.EntryType;
import net.sf.jabref.model.entry.InternalBibtexFields;
import net.sf.jabref.model.entry.MonthUtil;
import net.sf.jabref.model.entry.event.EntryEventSource;
import net.sf.jabref.model.entry.event.FieldChangedEvent;
import net.sf.jabref.model.strings.StringUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class BibDatabase {
    private static final Log LOGGER = LogFactory.getLog(BibDatabase.class);
    private final List<BibEntry> entries = Collections.synchronizedList(new ArrayList());
    private String preamble;
    private String epilog = "";
    private final Map<String, BibtexString> bibtexStrings = new ConcurrentHashMap<String, BibtexString>();
    private final DuplicationChecker duplicationChecker = new DuplicationChecker();
    private final Set<String> internalIDs = new HashSet<String>();
    private final EventBus eventBus = new EventBus();
    private String sharedDatabaseID;
    private static final Pattern RESOLVE_CONTENT_PATTERN = Pattern.compile(".*#[^#]+#.*");

    public BibDatabase() {
        this.eventBus.register(this.duplicationChecker);
        this.registerListener(new KeyChangeListener(this));
    }

    public int getEntryCount() {
        return this.entries.size();
    }

    public boolean hasEntries() {
        return !this.entries.isEmpty();
    }

    public synchronized EntrySorter getSorter(Comparator<BibEntry> comp) {
        return new EntrySorter(new ArrayList<BibEntry>(this.getEntries()), comp);
    }

    public boolean containsEntryWithId(String id) {
        return this.internalIDs.contains(id);
    }

    public List<BibEntry> getEntries() {
        return Collections.unmodifiableList(this.entries);
    }

    public Set<String> getAllVisibleFields() {
        TreeSet<String> allFields = new TreeSet<String>();
        for (BibEntry e : this.getEntries()) {
            allFields.addAll(e.getFieldNames());
        }
        return allFields.stream().filter(field -> !InternalBibtexFields.isInternalField(field)).collect(Collectors.toSet());
    }

    public synchronized Optional<BibEntry> getEntryByKey(String key) {
        for (BibEntry entry : this.entries) {
            if (!key.equals(entry.getCiteKeyOptional().orElse(null))) continue;
            return Optional.of(entry);
        }
        return Optional.empty();
    }

    public synchronized List<BibEntry> getEntriesByKey(String key) {
        ArrayList<BibEntry> result = new ArrayList<BibEntry>();
        for (BibEntry entry : this.entries) {
            entry.getCiteKeyOptional().ifPresent(entryKey -> {
                if (key.equals(entryKey)) {
                    result.add(entry);
                }
            });
        }
        return result;
    }

    public synchronized boolean insertEntry(BibEntry entry) throws KeyCollisionException {
        return this.insertEntry(entry, EntryEventSource.LOCAL);
    }

    public synchronized boolean insertEntry(BibEntry entry, EntryEventSource eventSource) throws KeyCollisionException {
        Objects.requireNonNull(entry);
        String id = entry.getId();
        if (this.containsEntryWithId(id)) {
            throw new KeyCollisionException("ID is already in use, please choose another");
        }
        this.internalIDs.add(id);
        this.entries.add(entry);
        entry.registerListener(this);
        this.eventBus.post(new EntryAddedEvent(entry, eventSource));
        return this.duplicationChecker.isDuplicateCiteKeyExisting(entry);
    }

    public synchronized void removeEntry(BibEntry toBeDeleted) {
        this.removeEntry(toBeDeleted, EntryEventSource.LOCAL);
    }

    public synchronized void removeEntry(BibEntry toBeDeleted, EntryEventSource eventSource) {
        Objects.requireNonNull(toBeDeleted);
        boolean anyRemoved = this.entries.removeIf(entry -> entry.getId().equals(toBeDeleted.getId()));
        if (anyRemoved) {
            this.internalIDs.remove(toBeDeleted.getId());
            this.eventBus.post(new EntryRemovedEvent(toBeDeleted, eventSource));
        }
    }

    public synchronized void setPreamble(String preamble) {
        this.preamble = preamble;
    }

    public synchronized Optional<String> getPreamble() {
        if (StringUtil.isBlank(this.preamble)) {
            return Optional.empty();
        }
        return Optional.of(this.preamble);
    }

    public synchronized void addString(BibtexString string) throws KeyCollisionException {
        if (this.hasStringLabel(string.getName())) {
            throw new KeyCollisionException("A string with that label already exists");
        }
        if (this.bibtexStrings.containsKey(string.getId())) {
            throw new KeyCollisionException("Duplicate BibTeX string id.");
        }
        this.bibtexStrings.put(string.getId(), string);
    }

    public void removeString(String id) {
        this.bibtexStrings.remove(id);
    }

    public Set<String> getStringKeySet() {
        return this.bibtexStrings.keySet();
    }

    public Collection<BibtexString> getStringValues() {
        return this.bibtexStrings.values();
    }

    public BibtexString getString(String id) {
        return this.bibtexStrings.get(id);
    }

    public int getStringCount() {
        return this.bibtexStrings.size();
    }

    public boolean hasNoStrings() {
        return this.bibtexStrings.isEmpty();
    }

    public void copyPreamble(BibDatabase database) {
        this.setPreamble(database.getPreamble().orElse(""));
    }

    public void copyStrings(BibDatabase database) {
        for (String key : database.getStringKeySet()) {
            BibtexString string = database.getString(key);
            this.addString(string);
        }
    }

    public synchronized boolean hasStringLabel(String label) {
        for (BibtexString value : this.bibtexStrings.values()) {
            if (!value.getName().equals(label)) continue;
            return true;
        }
        return false;
    }

    public String resolveForStrings(String content) {
        Objects.requireNonNull(content, "Content for resolveForStrings must not be null.");
        return this.resolveContent(content, new HashSet<String>());
    }

    public List<BibEntry> resolveForStrings(Collection<BibEntry> entriesToResolve, boolean inPlace) {
        Objects.requireNonNull(entriesToResolve, "entries must not be null.");
        ArrayList<BibEntry> results = new ArrayList<BibEntry>(entriesToResolve.size());
        for (BibEntry entry : entriesToResolve) {
            results.add(this.resolveForStrings(entry, inPlace));
        }
        return results;
    }

    public BibEntry resolveForStrings(BibEntry entry, boolean inPlace) {
        BibEntry resultingEntry = inPlace ? entry : (BibEntry)entry.clone();
        for (Map.Entry<String, String> field : resultingEntry.getFieldMap().entrySet()) {
            resultingEntry.setField(field.getKey(), this.resolveForStrings(field.getValue()));
        }
        return resultingEntry;
    }

    private String resolveString(String label, Set<String> usedIds) {
        for (BibtexString string : this.bibtexStrings.values()) {
            if (!string.getName().equalsIgnoreCase(label)) continue;
            if (usedIds.contains(string.getId())) {
                LOGGER.info("Stopped due to circular reference in strings: " + label);
                return label;
            }
            usedIds.add(string.getId());
            String result = string.getContent();
            result = this.resolveContent(result, usedIds);
            usedIds.remove(string.getId());
            return result;
        }
        MonthUtil.Month month = MonthUtil.getMonthByShortName(label);
        if (month.isValid()) {
            return month.fullName;
        }
        return null;
    }

    private String resolveContent(String result, Set<String> usedIds) {
        String res = result;
        if (RESOLVE_CONTENT_PATTERN.matcher(res).matches()) {
            int next;
            StringBuilder newRes = new StringBuilder();
            int piv = 0;
            while ((next = res.indexOf(35, piv)) >= 0) {
                int stringEnd;
                if (next > 0) {
                    newRes.append(res.substring(piv, next));
                }
                if ((stringEnd = res.indexOf(35, next + 1)) >= 0) {
                    String refLabel = res.substring(next + 1, stringEnd);
                    String resolved = this.resolveString(refLabel, usedIds);
                    if (resolved == null) {
                        newRes.append(res.substring(next, stringEnd + 1));
                    } else {
                        newRes.append(resolved);
                    }
                    piv = stringEnd + 1;
                    continue;
                }
                newRes.append(res.substring(next));
                piv = res.length();
                break;
            }
            if (piv < res.length() - 1) {
                newRes.append(res.substring(piv));
            }
            res = newRes.toString();
        }
        return res;
    }

    public static Optional<String> getResolvedField(String field, BibEntry entry, BibDatabase database) {
        Optional<BibEntry> referred;
        Optional<String> crossrefKey;
        Objects.requireNonNull(entry, "entry cannot be null");
        if ("entrytype".equals(field) || "bibtextype".equals(field)) {
            Optional<EntryType> entryType = EntryTypes.getType(entry.getType(), BibDatabaseMode.BIBLATEX);
            if (entryType.isPresent()) {
                return Optional.of(entryType.get().getName());
            }
            return Optional.of(StringUtil.capitalizeFirst(entry.getType()));
        }
        if ("bibtexkey".equals(field)) {
            return entry.getCiteKeyOptional();
        }
        Optional<String> result = entry.getFieldOrAlias(field);
        if (!result.isPresent() && database != null && (crossrefKey = entry.getField("crossref")).isPresent() && !crossrefKey.get().isEmpty() && (referred = database.getEntryByKey(crossrefKey.get())).isPresent()) {
            result = referred.get().getFieldOrAlias(field);
        }
        return result.map(resultText -> BibDatabase.getText(resultText, database));
    }

    @Deprecated
    public static String getText(String toResolve, BibDatabase database) {
        if (toResolve != null && database != null) {
            return database.resolveForStrings(toResolve);
        }
        return toResolve;
    }

    public void setEpilog(String epilog) {
        this.epilog = epilog;
    }

    public String getEpilog() {
        return this.epilog;
    }

    public void registerListener(Object listener) {
        this.eventBus.register(listener);
    }

    public void unregisterListener(Object listener) {
        this.eventBus.unregister(listener);
    }

    @Subscribe
    private void relayEntryChangeEvent(FieldChangedEvent event) {
        this.eventBus.post(event);
    }

    public Optional<BibEntry> getReferencedEntry(BibEntry entry) {
        return entry.getField("crossref").flatMap(this::getEntryByKey);
    }

    public Optional<String> getSharedDatabaseID() {
        return Optional.ofNullable(this.sharedDatabaseID);
    }

    public boolean isShared() {
        return this.getSharedDatabaseID().isPresent();
    }

    public void setSharedDatabaseID(String sharedDatabaseID) {
        this.sharedDatabaseID = sharedDatabaseID;
    }

    public void clearSharedDatabaseID() {
        this.sharedDatabaseID = null;
    }

    public String generateSharedDatabaseID() {
        this.sharedDatabaseID = new BigInteger(128, new SecureRandom()).toString(32);
        return this.sharedDatabaseID;
    }

    public DuplicationChecker getDuplicationChecker() {
        return this.duplicationChecker;
    }
}

