/*
 * Decompiled with CFR 0.152.
 */
package net.sf.jabref.logic.xmp;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.TreeSet;
import javax.xml.transform.TransformerException;
import net.sf.jabref.logic.TypedBibEntry;
import net.sf.jabref.logic.xmp.EncryptedPdfsNotSupportedException;
import net.sf.jabref.logic.xmp.XMPPreferences;
import net.sf.jabref.logic.xmp.XMPSchemaBibtex;
import net.sf.jabref.model.database.BibDatabase;
import net.sf.jabref.model.database.BibDatabaseMode;
import net.sf.jabref.model.entry.Author;
import net.sf.jabref.model.entry.AuthorList;
import net.sf.jabref.model.entry.BibEntry;
import net.sf.jabref.model.entry.MonthUtil;
import net.sf.jabref.model.strings.StringUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.jempbox.impl.DateConverter;
import org.apache.jempbox.impl.XMLUtil;
import org.apache.jempbox.xmp.XMPMetadata;
import org.apache.jempbox.xmp.XMPSchema;
import org.apache.jempbox.xmp.XMPSchemaDublinCore;
import org.apache.pdfbox.cos.COSBase;
import org.apache.pdfbox.cos.COSDictionary;
import org.apache.pdfbox.cos.COSName;
import org.apache.pdfbox.exceptions.COSVisitorException;
import org.apache.pdfbox.exceptions.CryptographyException;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDDocumentCatalog;
import org.apache.pdfbox.pdmodel.PDDocumentInformation;
import org.apache.pdfbox.pdmodel.common.PDMetadata;
import org.apache.pdfbox.pdmodel.encryption.BadSecurityHandlerException;
import org.apache.pdfbox.pdmodel.encryption.StandardDecryptionMaterial;
import org.w3c.dom.Document;

public class XMPUtil {
    private static final Log LOGGER = LogFactory.getLog(XMPUtil.class);

    public static List<BibEntry> readXMP(String filename, XMPPreferences xmpPreferences) throws IOException {
        return XMPUtil.readXMP(new File(filename), xmpPreferences);
    }

    public static void writeXMP(String filename, BibEntry entry, BibDatabase database, XMPPreferences xmpPreferences) throws IOException, TransformerException {
        XMPUtil.writeXMP(new File(filename), entry, database, xmpPreferences);
    }

    public static List<BibEntry> readXMP(File file, XMPPreferences xmpPreferences) throws IOException {
        List<BibEntry> result = Collections.emptyList();
        try (FileInputStream inputStream = new FileInputStream(file);){
            result = XMPUtil.readXMP(inputStream, xmpPreferences);
        }
        return result;
    }

    public static PDDocument loadWithAutomaticDecryption(InputStream inputStream) throws IOException {
        PDDocument doc = PDDocument.load(inputStream);
        if (doc.isEncrypted()) {
            StandardDecryptionMaterial sdm = new StandardDecryptionMaterial("");
            try {
                doc.openProtection(sdm);
            }
            catch (CryptographyException | BadSecurityHandlerException e) {
                LOGGER.error("Cannot handle encrypted PDF: " + e.getMessage());
                throw new EncryptedPdfsNotSupportedException();
            }
        }
        return doc;
    }

    public static List<BibEntry> readXMP(InputStream inputStream, XMPPreferences xmpPreferences) throws IOException {
        LinkedList<BibEntry> result = new LinkedList<BibEntry>();
        try (PDDocument document = XMPUtil.loadWithAutomaticDecryption(inputStream);){
            PDDocumentInformation documentInformation;
            Optional<BibEntry> entry;
            Optional<XMPMetadata> meta = XMPUtil.getXMPMetadata(document);
            if (meta.isPresent()) {
                Optional<BibEntry> entry2;
                List<XMPSchema> schemas = meta.get().getSchemasByNamespaceURI("http://jabref.sourceforge.net/bibteXMP/");
                for (XMPSchema schema : schemas) {
                    XMPSchemaBibtex bib = (XMPSchemaBibtex)schema;
                    entry2 = bib.getBibtexEntry();
                    if (((BibEntry)((Object)entry2)).getType() == null) {
                        ((BibEntry)((Object)entry2)).setType("misc");
                    }
                    result.add((BibEntry)((Object)entry2));
                }
                if (result.isEmpty()) {
                    schemas = meta.get().getSchemasByNamespaceURI("http://purl.org/dc/elements/1.1/");
                    for (XMPSchema schema : schemas) {
                        XMPSchemaDublinCore dc = (XMPSchemaDublinCore)schema;
                        entry2 = XMPUtil.getBibtexEntryFromDublinCore(dc, xmpPreferences);
                        if (!entry2.isPresent()) continue;
                        if (((BibEntry)entry2.get()).getType() == null) {
                            ((BibEntry)entry2.get()).setType("misc");
                        }
                        result.add((BibEntry)entry2.get());
                    }
                }
            }
            if (result.isEmpty() && (entry = XMPUtil.getBibtexEntryFromDocumentInformation(documentInformation = document.getDocumentInformation())).isPresent()) {
                result.add(entry.get());
            }
        }
        if (result.isEmpty()) {
            return Collections.emptyList();
        }
        return result;
    }

    public static Collection<BibEntry> readXMP(Path filePath, XMPPreferences xmpPreferences) throws IOException {
        return XMPUtil.readXMP(filePath.toFile(), xmpPreferences);
    }

    public static Optional<BibEntry> getBibtexEntryFromDocumentInformation(PDDocumentInformation di) {
        BibEntry entry = new BibEntry();
        entry.setType("misc");
        String s = di.getAuthor();
        if (s != null) {
            entry.setField("author", s);
        }
        if ((s = di.getTitle()) != null) {
            entry.setField("title", s);
        }
        if ((s = di.getKeywords()) != null) {
            entry.setField("keywords", s);
        }
        if ((s = di.getSubject()) != null) {
            entry.setField("abstract", s);
        }
        COSDictionary dict = di.getDictionary();
        for (Map.Entry<COSName, COSBase> o : dict.entrySet()) {
            String key = o.getKey().getName();
            if (!key.startsWith("bibtex/")) continue;
            String value = dict.getString(key);
            if ("entrytype".equals(key = key.substring("bibtex/".length()))) {
                entry.setType(value);
                continue;
            }
            entry.setField(key, value);
        }
        return entry.getFieldNames().isEmpty() ? Optional.empty() : Optional.of(entry);
    }

    public static Optional<BibEntry> getBibtexEntryFromDublinCore(XMPSchemaDublinCore dcSchema, XMPPreferences xmpPreferences) {
        List<String> l;
        List<String> subjects;
        List<String> relationships;
        List<String> publishers;
        String s;
        List<String> dates;
        List<String> creators;
        BibEntry entry = new BibEntry();
        List<String> contributors = dcSchema.getContributors();
        if (contributors != null && !contributors.isEmpty()) {
            entry.setField("editor", String.join((CharSequence)" and ", contributors));
        }
        if ((creators = dcSchema.getCreators()) != null && !creators.isEmpty()) {
            entry.setField("author", String.join((CharSequence)" and ", creators));
        }
        if ((dates = dcSchema.getSequenceList("dc:date")) != null && !dates.isEmpty()) {
            String date = dates.get(0).trim();
            Calendar c = null;
            try {
                c = DateConverter.toCalendar(date);
            }
            catch (IOException iOException) {
                // empty catch block
            }
            if (c != null) {
                entry.setField("year", String.valueOf(c.get(1)));
                if (date.length() > 4) {
                    entry.setField("month", MonthUtil.getMonthByIndex((int)c.get((int)2)).bibtexFormat);
                }
            }
        }
        if ((s = dcSchema.getDescription()) != null) {
            entry.setField("abstract", s);
        }
        if ((s = dcSchema.getIdentifier()) != null) {
            entry.setField("doi", s);
        }
        if ((publishers = dcSchema.getPublishers()) != null && !publishers.isEmpty()) {
            entry.setField("publisher", String.join((CharSequence)" and ", publishers));
        }
        if ((relationships = dcSchema.getRelationships()) != null) {
            for (String r : relationships) {
                int i;
                if (!r.startsWith("bibtex/") || (i = (r = r.substring("bibtex/".length())).indexOf(47)) == -1) continue;
                entry.setField(r.substring(0, i), r.substring(i + 1));
            }
        }
        if ((s = dcSchema.getRights()) != null) {
            entry.setField("rights", s);
        }
        if ((s = dcSchema.getSource()) != null) {
            entry.setField("source", s);
        }
        if ((subjects = dcSchema.getSubjects()) != null) {
            entry.addKeywords(subjects, xmpPreferences.getKeywordSeparator());
        }
        if ((s = dcSchema.getTitle()) != null) {
            entry.setField("title", s);
        }
        if ((l = dcSchema.getTypes()) != null && !l.isEmpty() && (s = l.get(0)) != null) {
            entry.setType(s);
        }
        return entry.getFieldNames().isEmpty() ? Optional.empty() : Optional.of(entry);
    }

    public static void writeXMP(File file, BibEntry entry, BibDatabase database, XMPPreferences xmpPreferences) throws IOException, TransformerException {
        LinkedList<BibEntry> l = new LinkedList<BibEntry>();
        l.add(entry);
        XMPUtil.writeXMP(file, l, database, true, xmpPreferences);
    }

    private static void toXMP(Collection<BibEntry> bibtexEntries, BibDatabase database, OutputStream outputStream, XMPPreferences xmpPreferences) throws IOException, TransformerException {
        Collection<BibEntry> resolvedEntries = database == null ? bibtexEntries : database.resolveForStrings(bibtexEntries, true);
        XMPMetadata x = new XMPMetadata();
        for (BibEntry e : resolvedEntries) {
            XMPSchemaBibtex schema = new XMPSchemaBibtex(x);
            x.addSchema(schema);
            schema.setBibtexEntry(e, xmpPreferences);
        }
        x.save(outputStream);
    }

    public static String toXMP(Collection<BibEntry> bibtexEntries, BibDatabase database, XMPPreferences xmpPreferences) throws TransformerException {
        try {
            ByteArrayOutputStream bs = new ByteArrayOutputStream();
            XMPUtil.toXMP(bibtexEntries, database, bs, xmpPreferences);
            return bs.toString();
        }
        catch (IOException e) {
            throw new TransformerException(e);
        }
    }

    private static Optional<XMPMetadata> readRawXMP(InputStream inputStream) throws IOException {
        try (PDDocument document = XMPUtil.loadWithAutomaticDecryption(inputStream);){
            Optional<XMPMetadata> optional = XMPUtil.getXMPMetadata(document);
            return optional;
        }
    }

    private static Optional<XMPMetadata> getXMPMetadata(PDDocument document) throws IOException {
        Document parseResult;
        PDDocumentCatalog catalog = document.getDocumentCatalog();
        PDMetadata metaRaw = catalog.getMetadata();
        if (metaRaw == null) {
            return Optional.empty();
        }
        try (InputStream is = metaRaw.createInputStream();){
            parseResult = XMLUtil.parse(is);
        }
        XMPMetadata meta = new XMPMetadata(parseResult);
        meta.addXMLNSMapping("http://jabref.sourceforge.net/bibteXMP/", XMPSchemaBibtex.class);
        return Optional.of(meta);
    }

    public static Optional<XMPMetadata> readRawXMP(File file) throws IOException {
        try (FileInputStream inputStream = new FileInputStream(file);){
            Optional<XMPMetadata> optional = XMPUtil.readRawXMP(inputStream);
            return optional;
        }
    }

    private static void writeToDCSchema(XMPSchemaDublinCore dcSchema, BibEntry entry, BibDatabase database, XMPPreferences xmpPreferences) {
        BibEntry resolvedEntry = database == null ? entry : database.resolveForStrings(entry, false);
        boolean useXmpPrivacyFilter = xmpPreferences.isUseXMPPrivacyFilter();
        TreeSet<String> filters = new TreeSet<String>(xmpPreferences.getXmpPrivacyFilter());
        for (Map.Entry<String, String> field : resolvedEntry.getFieldMap().entrySet()) {
            String o;
            AuthorList list;
            String authors;
            if (useXmpPrivacyFilter && filters.contains(field.getKey())) continue;
            if ("editor".equals(field.getKey())) {
                authors = field.getValue();
                list = AuthorList.parse(authors);
                for (Author author : list.getAuthors()) {
                    dcSchema.addContributor(author.getFirstLast(false));
                }
                continue;
            }
            if ("author".equals(field.getKey())) {
                authors = field.getValue();
                list = AuthorList.parse(authors);
                for (Author author : list.getAuthors()) {
                    dcSchema.addCreator(author.getFirstLast(false));
                }
                continue;
            }
            if ("month".equals(field.getKey())) continue;
            if ("year".equals(field.getKey())) {
                entry.getPublicationDate().ifPresent(publicationDate -> dcSchema.addSequenceValue("dc:date", (String)publicationDate));
                continue;
            }
            if ("abstract".equals(field.getKey())) {
                dcSchema.setDescription(field.getValue());
                continue;
            }
            if ("doi".equals(field.getKey())) {
                dcSchema.setIdentifier(field.getValue());
                continue;
            }
            if ("publisher".equals(field.getKey())) {
                dcSchema.addPublisher(field.getValue());
                continue;
            }
            if ("keywords".equals(field.getKey())) {
                String[] keywords;
                o = field.getValue();
                for (String keyword : keywords = o.split(",")) {
                    dcSchema.addSubject(keyword.trim());
                }
                continue;
            }
            if ("title".equals(field.getKey())) {
                dcSchema.setTitle(field.getValue());
                continue;
            }
            o = field.getValue();
            dcSchema.addRelation("bibtex/" + field.getKey() + '/' + o);
        }
        dcSchema.setFormat("application/pdf");
        TypedBibEntry typedEntry = new TypedBibEntry(entry, BibDatabaseMode.BIBTEX);
        String o = typedEntry.getTypeForDisplay();
        if (!o.isEmpty()) {
            dcSchema.addType(o);
        }
    }

    public static void writeDublinCore(PDDocument document, BibEntry entry, BibDatabase database, XMPPreferences xmpPreferences) throws IOException, TransformerException {
        ArrayList<BibEntry> entries = new ArrayList<BibEntry>();
        entries.add(entry);
        XMPUtil.writeDublinCore(document, entries, database, xmpPreferences);
    }

    private static void writeDublinCore(PDDocument document, Collection<BibEntry> entries, BibDatabase database, XMPPreferences xmpPreferences) throws IOException, TransformerException {
        Collection<BibEntry> resolvedEntries = database == null ? entries : database.resolveForStrings(entries, false);
        PDDocumentCatalog catalog = document.getDocumentCatalog();
        PDMetadata metaRaw = catalog.getMetadata();
        XMPMetadata meta = metaRaw == null ? new XMPMetadata() : new XMPMetadata(XMLUtil.parse(metaRaw.createInputStream()));
        List<XMPSchema> schemas = meta.getSchemasByNamespaceURI("http://purl.org/dc/elements/1.1/");
        for (XMPSchema schema : schemas) {
            schema.getElement().getParentNode().removeChild(schema.getElement());
        }
        for (BibEntry entry : resolvedEntries) {
            XMPSchemaDublinCore dcSchema = new XMPSchemaDublinCore(meta);
            XMPUtil.writeToDCSchema(dcSchema, entry, null, xmpPreferences);
            meta.addSchema(dcSchema);
        }
        ByteArrayOutputStream os = new ByteArrayOutputStream();
        meta.save(os);
        ByteArrayInputStream is = new ByteArrayInputStream(os.toByteArray());
        PDMetadata metadataStream = new PDMetadata(document, is, false);
        catalog.setMetadata(metadataStream);
    }

    private static void writeDocumentInformation(PDDocument document, BibEntry entry, BibDatabase database, XMPPreferences xmpPreferences) {
        PDDocumentInformation di = document.getDocumentInformation();
        BibEntry resolvedEntry = database == null ? entry : database.resolveForStrings(entry, false);
        boolean useXmpPrivacyFilter = xmpPreferences.isUseXMPPrivacyFilter();
        TreeSet<String> filters = new TreeSet<String>(xmpPreferences.getXmpPrivacyFilter());
        for (Map.Entry<String, String> field : resolvedEntry.getFieldMap().entrySet()) {
            String fieldName = field.getKey();
            String fieldContent = field.getValue();
            if (useXmpPrivacyFilter && filters.contains(fieldName)) {
                if ("author".equals(fieldName)) {
                    di.setAuthor(null);
                    continue;
                }
                if ("title".equals(fieldName)) {
                    di.setTitle(null);
                    continue;
                }
                if ("keywords".equals(fieldName)) {
                    di.setKeywords(null);
                    continue;
                }
                if ("abstract".equals(fieldName)) {
                    di.setSubject(null);
                    continue;
                }
                di.setCustomMetadataValue("bibtex/" + fieldName, null);
                continue;
            }
            if ("author".equals(fieldName)) {
                di.setAuthor(fieldContent);
                continue;
            }
            if ("title".equals(fieldName)) {
                di.setTitle(fieldContent);
                continue;
            }
            if ("keywords".equals(fieldName)) {
                di.setKeywords(fieldContent);
                continue;
            }
            if ("abstract".equals(fieldName)) {
                di.setSubject(fieldContent);
                continue;
            }
            di.setCustomMetadataValue("bibtex/" + fieldName, fieldContent);
        }
        di.setCustomMetadataValue("bibtex/entrytype", StringUtil.capitalizeFirst(resolvedEntry.getType()));
    }

    public static void writeXMP(File file, Collection<BibEntry> bibtexEntries, BibDatabase database, boolean writePDFInfo, XMPPreferences xmpPreferences) throws IOException, TransformerException {
        Collection<BibEntry> resolvedEntries = database == null ? bibtexEntries : database.resolveForStrings(bibtexEntries, false);
        try (PDDocument document = PDDocument.load(file.getAbsoluteFile());){
            PDDocumentCatalog catalog;
            PDMetadata metaRaw;
            if (document.isEncrypted()) {
                throw new EncryptedPdfsNotSupportedException();
            }
            if (writePDFInfo && resolvedEntries.size() == 1) {
                XMPUtil.writeDocumentInformation(document, resolvedEntries.iterator().next(), null, xmpPreferences);
                XMPUtil.writeDublinCore(document, resolvedEntries, null, xmpPreferences);
            }
            XMPMetadata meta = (metaRaw = (catalog = document.getDocumentCatalog()).getMetadata()) == null ? new XMPMetadata() : new XMPMetadata(XMLUtil.parse(metaRaw.createInputStream()));
            meta.addXMLNSMapping("http://jabref.sourceforge.net/bibteXMP/", XMPSchemaBibtex.class);
            List<XMPSchema> schemas = meta.getSchemasByNamespaceURI("http://jabref.sourceforge.net/bibteXMP/");
            for (XMPSchema schema : schemas) {
                XMPSchemaBibtex bib = (XMPSchemaBibtex)schema;
                bib.getElement().getParentNode().removeChild(bib.getElement());
            }
            for (BibEntry e : resolvedEntries) {
                XMPSchemaBibtex bibtex = new XMPSchemaBibtex(meta);
                meta.addSchema(bibtex);
                bibtex.setBibtexEntry(e, xmpPreferences);
            }
            ByteArrayOutputStream os = new ByteArrayOutputStream();
            meta.save(os);
            ByteArrayInputStream is = new ByteArrayInputStream(os.toByteArray());
            PDMetadata metadataStream = new PDMetadata(document, is, false);
            catalog.setMetadata(metadataStream);
            try {
                document.save(file.getAbsolutePath());
            }
            catch (COSVisitorException e) {
                LOGGER.debug("Could not write XMP metadata", e);
                throw new TransformerException("Could not write XMP metadata: " + e.getLocalizedMessage(), e);
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public static boolean hasMetadata(Path path, XMPPreferences xmpPreferences) {
        try (InputStream inputStream = Files.newInputStream(path, StandardOpenOption.READ);){
            boolean bl = XMPUtil.hasMetadata(inputStream, xmpPreferences);
            return bl;
        }
        catch (IOException e) {
            LOGGER.error("XMP reading failed", e);
            return false;
        }
    }

    public static boolean hasMetadata(InputStream inputStream, XMPPreferences xmpPreferences) {
        try {
            List<BibEntry> bibEntries = XMPUtil.readXMP(inputStream, xmpPreferences);
            return !bibEntries.isEmpty();
        }
        catch (EncryptedPdfsNotSupportedException ex) {
            LOGGER.info("Encryption not supported by XMPUtil");
            return false;
        }
        catch (IOException e) {
            LOGGER.error("XMP reading failed", e);
            return false;
        }
    }
}

