/*
 * Decompiled with CFR 0.152.
 */
package net.sf.jabref.logic.util;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.URL;
import java.net.URLConnection;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONArray;
import org.json.JSONObject;

public class Version {
    private static final Log LOGGER = LogFactory.getLog(Version.class);
    private static final Version UNKNOWN_VERSION = new Version();
    private static final Pattern VERSION_PATTERN = Pattern.compile("(?<major>\\d+)(\\.(?<minor>\\d+))?(\\.(?<patch>\\d+))?(?<stage>-alpha|-beta)?(?<dev>-?dev)?.*");
    public static final String JABREF_DOWNLOAD_URL = "https://downloads.jabref.org";
    private static final String JABREF_GITHUB_RELEASES = "https://api.github.com/repos/JabRef/JabRef/releases";
    private String fullVersion = "*unknown*";
    private int major = -1;
    private int minor = -1;
    private int patch = -1;
    private DevelopmentStage developmentStage = DevelopmentStage.UNKNOWN;
    private boolean isDevelopmentVersion;

    private Version() {
    }

    public static Version parse(String version) {
        if (version == null || "".equals(version) || version.equals("*unknown*") || "${version}".equals(version)) {
            return UNKNOWN_VERSION;
        }
        Version parsedVersion = new Version();
        parsedVersion.fullVersion = version;
        Matcher matcher = VERSION_PATTERN.matcher(version);
        if (matcher.find()) {
            try {
                parsedVersion.major = Integer.parseInt(matcher.group("major"));
                String minorString = matcher.group("minor");
                parsedVersion.minor = minorString == null ? 0 : Integer.parseInt(minorString);
                String patchString = matcher.group("patch");
                parsedVersion.patch = patchString == null ? 0 : Integer.parseInt(patchString);
                String versionStageString = matcher.group("stage");
                parsedVersion.developmentStage = versionStageString == null ? DevelopmentStage.STABLE : DevelopmentStage.parse(versionStageString);
                parsedVersion.isDevelopmentVersion = matcher.group("dev") != null;
            }
            catch (NumberFormatException e) {
                LOGGER.warn("Invalid version string used: " + version, e);
                return UNKNOWN_VERSION;
            }
            catch (IllegalArgumentException e) {
                LOGGER.warn("Invalid version pattern is used", e);
                return UNKNOWN_VERSION;
            }
        } else {
            LOGGER.warn("Version could not be recognized by the pattern");
            return UNKNOWN_VERSION;
        }
        return parsedVersion;
    }

    public static List<Version> getAllAvailableVersions() throws IOException {
        URLConnection connection = new URL(JABREF_GITHUB_RELEASES).openConnection();
        connection.setRequestProperty("Accept-Charset", "UTF-8");
        BufferedReader rd = new BufferedReader(new InputStreamReader(connection.getInputStream()));
        ArrayList<Version> versions = new ArrayList<Version>();
        JSONArray objects = new JSONArray(rd.readLine());
        for (int i = 0; i < objects.length(); ++i) {
            JSONObject jsonObject = objects.getJSONObject(i);
            Version version = Version.parse(jsonObject.getString("tag_name").replaceFirst("v", ""));
            versions.add(version);
        }
        return versions;
    }

    public boolean isNewerThan(Version otherVersion) {
        Objects.requireNonNull(otherVersion);
        if (Objects.equals(this, otherVersion)) {
            return false;
        }
        if (this.getFullVersion().equals("*unknown*")) {
            return false;
        }
        if (otherVersion.getFullVersion().equals("*unknown*")) {
            return false;
        }
        if (this.getMajor() > otherVersion.getMajor()) {
            return true;
        }
        if (this.getMajor() == otherVersion.getMajor()) {
            if (this.getMinor() > otherVersion.getMinor()) {
                return true;
            }
            if (this.getMinor() == otherVersion.getMinor()) {
                if (this.getPatch() > otherVersion.getPatch()) {
                    return true;
                }
                if (this.getPatch() == otherVersion.getPatch()) {
                    return this.developmentStage.isMoreStableThan(otherVersion.developmentStage);
                }
            }
        }
        return false;
    }

    public Optional<Version> shouldBeUpdatedTo(List<Version> availableVersions) {
        Optional<Version> newerVersion = Optional.empty();
        for (Version version : availableVersions) {
            if (!this.shouldBeUpdatedTo(version) || newerVersion.isPresent() && !version.isNewerThan(newerVersion.get())) continue;
            newerVersion = Optional.of(version);
        }
        return newerVersion;
    }

    public boolean shouldBeUpdatedTo(Version otherVersion) {
        if (this.developmentStage == DevelopmentStage.STABLE && otherVersion.developmentStage != DevelopmentStage.STABLE) {
            return false;
        }
        return otherVersion.isNewerThan(this);
    }

    public String getFullVersion() {
        return this.fullVersion;
    }

    public int getMajor() {
        return this.major;
    }

    public int getMinor() {
        return this.minor;
    }

    public int getPatch() {
        return this.patch;
    }

    public boolean isDevelopmentVersion() {
        return this.isDevelopmentVersion;
    }

    public String getChangelogUrl() {
        if (this.isDevelopmentVersion) {
            return "https://github.com/JabRef/jabref/blob/master/CHANGELOG.md#unreleased";
        }
        StringBuilder changelogLink = new StringBuilder().append("https://github.com/JabRef/jabref/blob/v").append(this.getMajor()).append(".").append(this.getMinor());
        if (this.getPatch() != 0) {
            changelogLink.append(".").append(this.getPatch());
        }
        changelogLink.append(this.developmentStage.stage).append("/CHANGELOG.md");
        return changelogLink.toString();
    }

    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }
        if (!(other instanceof Version)) {
            return false;
        }
        return this.getFullVersion().equals(((Version)other).getFullVersion());
    }

    public int hashCode() {
        return this.getFullVersion().hashCode();
    }

    public String toString() {
        return this.getFullVersion();
    }

    public static enum DevelopmentStage {
        UNKNOWN("", 0),
        ALPHA("-alpha", 1),
        BETA("-beta", 2),
        STABLE("", 3);

        private final int stability;
        private final String stage;

        private DevelopmentStage(String stage, int stability) {
            this.stage = stage;
            this.stability = stability;
        }

        public static DevelopmentStage parse(String stage) {
            if (stage == null) {
                LOGGER.warn("The stage cannot be null");
                return UNKNOWN;
            }
            if (stage.equals(DevelopmentStage.STABLE.stage)) {
                return STABLE;
            }
            if (stage.equals(DevelopmentStage.ALPHA.stage)) {
                return ALPHA;
            }
            if (stage.equals(DevelopmentStage.BETA.stage)) {
                return BETA;
            }
            LOGGER.warn("Unknown development stage: " + stage);
            return UNKNOWN;
        }

        public boolean isMoreStableThan(DevelopmentStage otherStage) {
            return this.stability > otherStage.stability;
        }
    }
}

