/*
 * Decompiled with CFR 0.152.
 */
package net.sf.jabref.logic.importer.fetcher;

import java.io.IOException;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import net.sf.jabref.logic.TypedBibEntry;
import net.sf.jabref.logic.help.HelpFile;
import net.sf.jabref.logic.importer.FetcherException;
import net.sf.jabref.logic.importer.FulltextFetcher;
import net.sf.jabref.logic.importer.IdBasedFetcher;
import net.sf.jabref.logic.importer.ImportFormatPreferences;
import net.sf.jabref.logic.importer.SearchBasedFetcher;
import net.sf.jabref.logic.importer.util.OAI2Handler;
import net.sf.jabref.logic.util.DOI;
import net.sf.jabref.logic.util.io.XMLUtil;
import net.sf.jabref.model.database.BibDatabaseMode;
import net.sf.jabref.model.entry.BibEntry;
import net.sf.jabref.model.entry.BibtexEntryTypes;
import net.sf.jabref.model.entry.ParsedFileField;
import net.sf.jabref.model.strings.StringUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.http.client.utils.URIBuilder;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

public class ArXiv
implements FulltextFetcher,
SearchBasedFetcher,
IdBasedFetcher {
    private static final Log LOGGER = LogFactory.getLog(ArXiv.class);
    private static final String API_URL = "http://export.arxiv.org/api/query";
    private final ImportFormatPreferences importFormatPreferences;

    public ArXiv(ImportFormatPreferences importFormatPreferences) {
        this.importFormatPreferences = importFormatPreferences;
    }

    @Override
    public Optional<URL> findFullText(BibEntry entry) throws IOException {
        Optional doi;
        Objects.requireNonNull(entry);
        Optional<String> identifier = entry.getField("eprint");
        if (StringUtil.isNotBlank(identifier)) {
            try {
                Optional<URL> pdfUrl = this.searchForEntryById(identifier.get()).flatMap(ArXivEntry::getPdfUrl);
                if (pdfUrl.isPresent()) {
                    LOGGER.info("Fulltext PDF found @ arXiv.");
                    return pdfUrl;
                }
            }
            catch (FetcherException e) {
                LOGGER.warn("arXiv eprint API request failed", e);
            }
        }
        if ((doi = entry.getField("doi").flatMap(DOI::build)).isPresent()) {
            String doiString = ((DOI)doi.get()).getDOI();
            try {
                Optional<URL> pdfUrl = this.searchForEntry("doi:" + doiString).flatMap(ArXivEntry::getPdfUrl);
                if (pdfUrl.isPresent()) {
                    LOGGER.info("Fulltext PDF found @ arXiv.");
                    return pdfUrl;
                }
            }
            catch (FetcherException e) {
                LOGGER.warn("arXiv DOI API request failed", e);
            }
        }
        return Optional.empty();
    }

    private Optional<ArXivEntry> searchForEntry(String searchQuery) throws FetcherException {
        List<ArXivEntry> entries = this.queryApi(searchQuery, Collections.emptyList(), 0, 1);
        if (entries.size() == 1) {
            return Optional.of(entries.get(0));
        }
        return Optional.empty();
    }

    private Optional<ArXivEntry> searchForEntryById(String identifier) throws FetcherException {
        List<ArXivEntry> entries = this.queryApi("", Collections.singletonList(identifier), 0, 1);
        if (entries.size() == 1) {
            return Optional.of(entries.get(0));
        }
        return Optional.empty();
    }

    private List<ArXivEntry> searchForEntries(String searchQuery) throws FetcherException {
        return this.queryApi(searchQuery, Collections.emptyList(), 0, 10);
    }

    private List<ArXivEntry> queryApi(String searchQuery, List<String> ids, int start, int maxResults) throws FetcherException {
        Document result = this.callApi(searchQuery, ids, start, maxResults);
        List<Node> entries = XMLUtil.asList(result.getElementsByTagName("entry"));
        return entries.stream().map(ArXivEntry::new).collect(Collectors.toList());
    }

    private Document callApi(String searchQuery, List<String> ids, int start, int maxResults) throws FetcherException {
        if (maxResults > 2000) {
            throw new IllegalArgumentException("The arXiv API limits the number of maximal results to be 2000");
        }
        try {
            URIBuilder uriBuilder = new URIBuilder(API_URL);
            if (StringUtil.isNotBlank(searchQuery)) {
                uriBuilder.addParameter("search_query", StringUtil.stripAccents(searchQuery));
            }
            if (!ids.isEmpty()) {
                uriBuilder.addParameter("id_list", String.join((CharSequence)",", ids));
            }
            uriBuilder.addParameter("start", String.valueOf(start));
            uriBuilder.addParameter("max_results", String.valueOf(maxResults));
            URL url = uriBuilder.build().toURL();
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            HttpURLConnection connection = (HttpURLConnection)url.openConnection();
            if (connection.getResponseCode() == 400) {
                throw this.getException(builder.parse(connection.getErrorStream()));
            }
            return builder.parse(connection.getInputStream());
        }
        catch (IOException | URISyntaxException | ParserConfigurationException | SAXException exception) {
            throw new FetcherException("arXiv API request failed", exception);
        }
    }

    private FetcherException getException(Document error) {
        Node node;
        Optional<String> id;
        Boolean isError;
        List<Node> entries = XMLUtil.asList(error.getElementsByTagName("entry"));
        if (entries.size() == 1 && (isError = (id = XMLUtil.getNodeContent(node = entries.get(0), "id")).map(idContent -> idContent.startsWith("http://arxiv.org/api/errors")).orElse(false)).booleanValue()) {
            String errorMessage = XMLUtil.getNodeContent(node, "summary").orElse("Unknown error");
            return new FetcherException(errorMessage);
        }
        return new FetcherException("arXiv API request failed");
    }

    @Override
    public String getName() {
        return "ArXiv";
    }

    @Override
    public HelpFile getHelpPage() {
        return HelpFile.FETCHER_OAI2_ARXIV;
    }

    @Override
    public List<BibEntry> performSearch(String query) throws FetcherException {
        return this.searchForEntries(query).stream().map(arXivEntry -> arXivEntry.toBibEntry(this.importFormatPreferences.getKeywordSeparator())).collect(Collectors.toList());
    }

    @Override
    public Optional<BibEntry> performSearchById(String identifier) throws FetcherException {
        return this.searchForEntryById(identifier).map(arXivEntry -> arXivEntry.toBibEntry(this.importFormatPreferences.getKeywordSeparator()));
    }

    private static class ArXivEntry {
        private final Optional<String> title;
        private final Optional<String> urlAbstractPage;
        private final Optional<String> publishedDate;
        private final Optional<String> abstractText;
        private final List<String> authorNames;
        private final List<String> categories;
        private final Optional<URL> pdfUrl;
        private final Optional<String> doi;
        private final Optional<String> journalReferenceText;
        private final Optional<String> primaryCategory;

        public ArXivEntry(Node item) {
            this.title = XMLUtil.getNodeContent(item, "title").map(OAI2Handler::correctLineBreaks);
            this.urlAbstractPage = XMLUtil.getNodeContent(item, "id");
            this.publishedDate = XMLUtil.getNodeContent(item, "published");
            this.abstractText = XMLUtil.getNodeContent(item, "summary").map(OAI2Handler::correctLineBreaks).map(String::trim);
            this.authorNames = new ArrayList<String>();
            for (Node authorNode : XMLUtil.getNodesByName(item, "author")) {
                Optional<String> authorName = XMLUtil.getNodeContent(authorNode, "name").map(String::trim);
                authorName.ifPresent(this.authorNames::add);
            }
            this.categories = new ArrayList<String>();
            for (Node categoryNode : XMLUtil.getNodesByName(item, "category")) {
                Optional<String> category = XMLUtil.getAttributeContent(categoryNode, "term");
                category.ifPresent(this.categories::add);
            }
            Optional<Object> pdfUrlParsed = Optional.empty();
            for (Node linkNode : XMLUtil.getNodesByName(item, "link")) {
                Optional<String> linkTitle = XMLUtil.getAttributeContent(linkNode, "title");
                if (!linkTitle.equals(Optional.of("pdf"))) continue;
                pdfUrlParsed = XMLUtil.getAttributeContent(linkNode, "href").map(url -> {
                    try {
                        return new URL((String)url);
                    }
                    catch (MalformedURLException e) {
                        return null;
                    }
                });
            }
            this.pdfUrl = pdfUrlParsed;
            this.doi = XMLUtil.getNodeContent(item, "arxiv:doi");
            this.journalReferenceText = XMLUtil.getNodeContent(item, "arxiv:journal_ref");
            this.primaryCategory = XMLUtil.getNode(item, "arxiv:primary_category").flatMap(node -> XMLUtil.getAttributeContent(node, "term"));
        }

        public Optional<URL> getPdfUrl() {
            return this.pdfUrl;
        }

        public Optional<String> getId() {
            String prefix = "http://arxiv.org/abs/";
            return this.urlAbstractPage.map(abstractUrl -> {
                if (abstractUrl.startsWith(prefix)) {
                    return abstractUrl.substring(prefix.length());
                }
                return abstractUrl;
            });
        }

        public Optional<String> getDate() {
            return this.publishedDate.map(date -> {
                if (date.length() < 10) {
                    return null;
                }
                return date.substring(0, 10);
            });
        }

        public BibEntry toBibEntry(Character keywordDelimiter) {
            BibEntry bibEntry = new BibEntry();
            bibEntry.setType(BibtexEntryTypes.ARTICLE);
            bibEntry.setField("eprinttype", "arXiv");
            bibEntry.setField("author", String.join((CharSequence)" and ", this.authorNames));
            bibEntry.addKeywords(this.categories, keywordDelimiter);
            this.getId().ifPresent(id -> bibEntry.setField("eprint", (String)id));
            this.title.ifPresent(titleContent -> bibEntry.setField("title", (String)titleContent));
            this.doi.ifPresent(doiContent -> bibEntry.setField("doi", (String)doiContent));
            this.abstractText.ifPresent(abstractContent -> bibEntry.setField("abstract", (String)abstractContent));
            this.getDate().ifPresent(date -> bibEntry.setField("date", (String)date));
            this.primaryCategory.ifPresent(category -> bibEntry.setField("eprintclass", (String)category));
            this.journalReferenceText.ifPresent(journal -> bibEntry.setField("journaltitle", (String)journal));
            this.getPdfUrl().ifPresent(url -> new TypedBibEntry(bibEntry, BibDatabaseMode.BIBLATEX).setFiles(Collections.singletonList(new ParsedFileField("online", (URL)url, "PDF"))));
            return bibEntry;
        }
    }
}

