/*
 * Decompiled with CFR 0.152.
 */
package net.sf.jabref.logic.autosaveandbackup;

import com.google.common.eventbus.Subscribe;
import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.util.HashSet;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import net.sf.jabref.logic.exporter.BibtexDatabaseWriter;
import net.sf.jabref.logic.exporter.FileSaveSession;
import net.sf.jabref.logic.exporter.SaveException;
import net.sf.jabref.logic.exporter.SavePreferences;
import net.sf.jabref.logic.util.io.FileUtil;
import net.sf.jabref.model.database.BibDatabaseContext;
import net.sf.jabref.model.database.event.BibDatabaseContextChangedEvent;
import net.sf.jabref.preferences.JabRefPreferences;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class BackupManager {
    private static final Log LOGGER = LogFactory.getLog(BackupManager.class);
    private static final String BACKUP_EXTENSION = ".sav";
    private static Set<BackupManager> runningInstances = new HashSet<BackupManager>();
    private final BibDatabaseContext bibDatabaseContext;
    private final JabRefPreferences preferences;
    private final BlockingQueue<Runnable> workerQueue;
    private final ExecutorService executor;
    private final Charset charset;
    private Path originalPath;
    private Path backupPath;
    private final Runnable backupTask = new Runnable(){

        @Override
        public void run() {
            try {
                SavePreferences prefs = SavePreferences.loadForSaveFromPreferences(BackupManager.this.preferences).withEncoding(BackupManager.this.charset).withMakeBackup(false);
                ((FileSaveSession)new BibtexDatabaseWriter<FileSaveSession>(FileSaveSession::new).saveDatabase(BackupManager.this.bibDatabaseContext, prefs)).commit(BackupManager.this.backupPath);
            }
            catch (SaveException e) {
                LOGGER.error("Error while saving file.", e);
            }
        }
    };

    private BackupManager(BibDatabaseContext bibDatabaseContext) {
        this.bibDatabaseContext = bibDatabaseContext;
        this.preferences = JabRefPreferences.getInstance();
        this.workerQueue = new ArrayBlockingQueue<Runnable>(1);
        this.executor = new ThreadPoolExecutor(1, 1, 0L, TimeUnit.SECONDS, this.workerQueue);
        this.charset = bibDatabaseContext.getMetaData().getEncoding().orElse(this.preferences.getDefaultEncoding());
    }

    @Subscribe
    public synchronized void listen(BibDatabaseContextChangedEvent event) {
        this.startBackupTask();
    }

    private void startBackupTask() {
        try {
            this.executor.submit(this.backupTask);
        }
        catch (RejectedExecutionException e) {
            LOGGER.debug("Rejecting while another backup process is already running.");
        }
    }

    private void shutdown() {
        this.bibDatabaseContext.getDatabase().unregisterListener(this);
        this.bibDatabaseContext.getMetaData().unregisterListener(this);
        this.executor.shutdown();
        try {
            if (Files.exists(this.backupPath, new LinkOption[0]) && !Files.isDirectory(this.backupPath, new LinkOption[0])) {
                Files.delete(this.backupPath);
            }
        }
        catch (IOException e) {
            LOGGER.error("Error while deleting the backup file.", e);
        }
    }

    static Path getBackupPath(Path originalPath) {
        return FileUtil.addExtension(originalPath, BACKUP_EXTENSION);
    }

    public static BackupManager start(BibDatabaseContext bibDatabaseContext) {
        BackupManager backupManager = new BackupManager(bibDatabaseContext);
        Optional<File> originalFile = bibDatabaseContext.getDatabaseFile();
        if (originalFile.isPresent()) {
            backupManager.originalPath = originalFile.get().toPath();
            backupManager.backupPath = BackupManager.getBackupPath(backupManager.originalPath);
            backupManager.startBackupTask();
            bibDatabaseContext.getDatabase().registerListener(backupManager);
            bibDatabaseContext.getMetaData().registerListener(backupManager);
            runningInstances.add(backupManager);
        }
        return backupManager;
    }

    public static void shutdown(BibDatabaseContext bibDatabaseContext) {
        runningInstances.stream().filter(instance -> instance.bibDatabaseContext == bibDatabaseContext).findAny().ifPresent(instance -> {
            instance.shutdown();
            runningInstances.remove(instance);
        });
    }

    public static boolean checkForBackupFile(Path originalPath) {
        Path backupPath = BackupManager.getBackupPath(originalPath);
        return Files.exists(backupPath, new LinkOption[0]) && !Files.isDirectory(backupPath, new LinkOption[0]);
    }

    public static void restoreBackup(Path originalPath) {
        Path backupPath = BackupManager.getBackupPath(originalPath);
        try {
            Files.copy(backupPath, originalPath, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            LOGGER.error("Error while restoring the backup file.", e);
        }
    }
}

