/*
 * Decompiled with CFR 0.152.
 */
package net.sf.jabref.collab;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.util.HashMap;
import java.util.Map;
import net.sf.jabref.collab.FileUpdateListener;
import net.sf.jabref.logic.util.io.FileUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

public class FileUpdateMonitor
implements Runnable {
    private static final Log LOGGER = LogFactory.getLog(FileUpdateMonitor.class);
    private static final int WAIT = 4000;
    private int numberOfUpdateListener;
    private final Map<String, Entry> entries = new HashMap<String, Entry>();

    @Override
    public void run() {
        while (true) {
            for (Entry e : this.entries.values()) {
                try {
                    if (!e.hasBeenUpdated()) continue;
                    e.notifyListener();
                }
                catch (IOException ex) {
                    e.notifyFileRemoved();
                }
            }
            try {
                Thread.sleep(4000L);
            }
            catch (InterruptedException ex) {
                LOGGER.debug("FileUpdateMonitor has been interrupted. Terminating...", ex);
                return;
            }
        }
    }

    public String addUpdateListener(FileUpdateListener ul, File file) throws IOException {
        if (!file.exists()) {
            throw new IOException("File not found");
        }
        ++this.numberOfUpdateListener;
        String key = String.valueOf(this.numberOfUpdateListener);
        this.entries.put(key, new Entry(ul, file.toPath()));
        return key;
    }

    public boolean hasBeenModified(String handle) {
        Entry entry = this.entries.get(handle);
        if (entry == null) {
            return false;
        }
        try {
            return entry.hasBeenUpdated();
        }
        catch (IOException ex) {
            return false;
        }
    }

    public void perturbTimestamp(String handle) {
        Entry entry = this.entries.get(handle);
        if (entry != null) {
            entry.decreaseTimeStamp();
        }
    }

    public void removeUpdateListener(String handle) {
        this.entries.remove(handle);
    }

    public void updateTimeStamp(String key) {
        Entry entry = this.entries.get(key);
        if (entry != null) {
            try {
                entry.updateTimeStamp();
            }
            catch (IOException e) {
                LOGGER.error("Couldn't update timestamp", e);
            }
        }
    }

    public Path getTempFile(String key) throws IllegalArgumentException {
        Entry entry = this.entries.get(key);
        if (entry == null) {
            throw new IllegalArgumentException("Entry not found");
        }
        return entry.getTmpFile();
    }

    private static synchronized Path getTempFile() {
        Path temporaryFile = null;
        try {
            temporaryFile = Files.createTempFile("jabref", null, new FileAttribute[0]);
            temporaryFile.toFile().deleteOnExit();
        }
        catch (IOException ex) {
            LOGGER.warn("Could not create temporary file.", ex);
        }
        return temporaryFile;
    }

    static class Entry {
        private final FileUpdateListener listener;
        private final Path file;
        private final Path tmpFile;
        private long timeStamp;
        private long fileSize;

        public Entry(FileUpdateListener ul, Path f) throws IOException {
            this.listener = ul;
            this.file = f;
            this.timeStamp = Files.getLastModifiedTime(this.file, new LinkOption[0]).toMillis();
            this.fileSize = Files.size(this.file);
            this.tmpFile = FileUpdateMonitor.getTempFile();
            if (this.tmpFile != null) {
                this.tmpFile.toFile().deleteOnExit();
                this.copy();
            }
        }

        public boolean hasBeenUpdated() throws IOException {
            long modified = Files.getLastModifiedTime(this.file, new LinkOption[0]).toMillis();
            if (modified == 0L) {
                throw new IOException("File deleted");
            }
            long fileSizeNow = Files.size(this.file);
            return this.timeStamp != modified || this.fileSize != fileSizeNow;
        }

        public void updateTimeStamp() throws IOException {
            this.timeStamp = Files.getLastModifiedTime(this.file, new LinkOption[0]).toMillis();
            if (this.timeStamp == 0L) {
                this.notifyFileRemoved();
            }
            this.fileSize = Files.size(this.file);
            this.copy();
        }

        public boolean copy() {
            boolean res = FileUtil.copyFile(this.file, this.tmpFile, true);
            return res;
        }

        public void notifyListener() throws IOException {
            this.timeStamp = Files.getLastModifiedTime(this.file, new LinkOption[0]).toMillis();
            this.fileSize = Files.size(this.file);
            this.listener.fileUpdated();
        }

        public void notifyFileRemoved() {
            this.listener.fileRemoved();
        }

        public Path getTmpFile() {
            return this.tmpFile;
        }

        public void decreaseTimeStamp() {
            --this.timeStamp;
        }
    }
}

